/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

var importSettings = require('./importSettings');
var jiraImportContext = require('./importContext');
var entities = require('@jetbrains/youtrack-scripting-api/entities');
var transaction = require('@jetbrains/youtrack-scripting-api-import/transaction');
var environment = require('@jetbrains/youtrack-scripting-api/environment');
var entityConverters = require('./entityConverters');
var jira = require('./jira');

// retrieve issue link types
entities.IssueLinkPrototype.all.forEach(function(it) {
  it.jiraId && (jiraImportContext.linkTypes.byJiraId[it.jiraId] = it);
  // name uniqueness in yt uses case insensitive comparison
  jiraImportContext.linkTypes.byTypeName[it.name.toLowerCase()] = it;
  it.localizedName && (jiraImportContext.linkTypes.byTypeName[it.localizedName.toLowerCase()] = it);
  it.sourceToTarget && (jiraImportContext.linkTypes.byLinkName[it.sourceToTarget.toLowerCase()] = it);
  it.localizedSourceToTarget && (jiraImportContext.linkTypes.byLinkName[it.localizedSourceToTarget.toLowerCase()] = it);
  it.targetToSource && (jiraImportContext.linkTypes.byLinkName[it.targetToSource.toLowerCase()] = it);
  it.localizedTargetToSource && (jiraImportContext.linkTypes.byLinkName[it.localizedTargetToSource.toLowerCase()] = it);
});
jiraImportContext.jiraClient.getLinkTypes(jiraImportContext.networkFailureHandler)['issueLinkTypes'].forEach(entityConverters.linkType);
transaction.flush();
console.info('Imported issue link types');

// retrieve jira field schema
jiraImportContext.jiraClient.getFields(function (fields) {
  fields.forEach(jiraImportContext.fieldSchema.visitFieldPrototype);
}, jiraImportContext.networkFailureHandler);
console.info('Loaded jira field schema');

var singleIssue = importSettings.issueId && jiraImportContext.jiraClient.getIssue(importSettings.issueId);
// retrieve field schema for each project specified in import settings
if (singleIssue) {
  jiraImportContext.fieldSchema.visitProject(jiraImportContext.jiraClient.getProject(singleIssue.fields.project.key, jiraImportContext.networkFailureHandler));
} else {
  jiraImportContext.jiraClient.getProjects(function (jiraProjects) {
    jiraProjects.filter(function (jiraProject) {
      var jiraProjectKey = jiraProject.key;
      if (importSettings.projects && importSettings.projects.length > 0 && importSettings.projects.indexOf(jiraProjectKey) === -1) {
        console.info('Skipping the project with key ' + jiraProjectKey);
        return false;
      }
      return true;
    }).forEach(jiraImportContext.fieldSchema.visitProject);
  }, jiraImportContext.networkFailureHandler);
}

// import projects
Object.keys(jiraImportContext.fieldSchema.projects).forEach(entityConverters.project);
environment.status.setOutput('stage', 'Projects created');
transaction.flush();

var currentTime = jiraImportContext.jiraClient.getServerInfo(jiraImportContext.networkFailureHandler).serverTime;
var importFinish = jiraImportContext.timeInterval.to = jira.roundTimestampUpToMinutes(currentTime);
jiraImportContext.userTimeZone = (jiraImportContext.jiraClient.getCurrentUser(jiraImportContext.networkFailureHandler) || {}).timeZone;
// import issues
var processIssuesForProject = function (projectKey) {
  var importStart = jiraImportContext.timeInterval.from[projectKey] = environment.status.getOutput(projectKey + '.importedTill');
  console.info('Importing issues only updated from "' + importStart + '" till "' + importFinish + '"');
  var lastProcessedIssueId = environment.status.getOutput(projectKey + '.lastProcessedIssueId');
  console.info('Starting to import project ' + projectKey + ' issues from id "' + lastProcessedIssueId + '"');

  var from = jira.toSearchTimeString(importStart, jiraImportContext.userTimeZone);
  var to = jira.toSearchTimeString(importFinish, jiraImportContext.userTimeZone);
  jiraImportContext.jiraClient.getIssues(projectKey, from, to, lastProcessedIssueId, importSettings.issuesBulkSize, function (jiraIssues) {
    if (jiraIssues.length) {
      jiraIssues.forEach(function (jiraIssue) { // updating schema
        entityConverters.issue(jiraIssue, true);
      });
      transaction.flush();
      jiraIssues.forEach(function (jiraIssue) { // importing issues
        entityConverters.issue(jiraIssue, false);
      });
      environment.status.setOutput(projectKey + '.lastProcessedIssueId', jiraIssues[jiraIssues.length - 1].id);
    }
    var progress = 'Importing issues from project ' + projectKey + ' (issues added: ' + jiraImportContext.progress.issuesAdded +
      ', issues updated: ' + jiraImportContext.progress.issuesUpdated + ')';
    environment.status.setOutput('stage', progress);
    transaction.flush();
    console.info(progress);
    return true;
  }, jiraImportContext.networkFailureHandler);

  environment.status.setOutput(projectKey + '.lastProcessedIssueId', null);
  environment.status.setOutput(projectKey + '.importedTill', importFinish);
  environment.status.setOutput('stage', projectKey + ' project import successfully finished. Issues imported till ' + importFinish);
  transaction.flush();
};
if (singleIssue) {
  // when importing a single issue, ignore previous import attempts results
  console.info('Importing a single issue: ' + singleIssue.key);
  entityConverters.issue(singleIssue, true);
  transaction.flush();
  entityConverters.issue(singleIssue, false);
  transaction.flush();
} else {
  Object.keys(jiraImportContext.fieldSchema.projects).forEach(processIssuesForProject);
  environment.status.setOutput('stage', 'Import successfully finished. Issues imported till ' + importFinish);
  transaction.flush();
}
