/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

/**
 * This module contains functions that let you work with dates and times.
 *
 * @module @jetbrains/youtrack-scripting-api/date-time
 */

var utils = Packages.jetbrains.youtrack.scripts.sandbox.Utils;
var Period = Packages.org.joda.time.Period;
var formatter = new Packages.org.joda.time.format.PeriodFormatterBuilder().rejectSignedValues(true).appendWeeks().appendSuffix('w').appendDays().appendSuffix('d').appendHours().appendSuffix('h').appendMinutes().appendSuffix('m').toFormatter();

/**
 * Parses a string representation of a date to return a Unix timestamp.
 * Use this method instead of the Date.parse() method from JavaScript.
 * For a detailed explanation, refer to the {@link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Date/parse|MDN documentation}.
 *
 * @param {string} dateTimeString The string representation of a date.
 *
 * @param {(string|string[])} [formats] A date format that possibly matches the dateTimeString or an array of formats.
 * If an array is provided, the formats are applied sequentially until the dateTimeString is parsed successfully.
 * If no value is specified, the date format is supplied by the system.
 * For actions that are attributed to the current user, the date format setting from the profile for the current user is applied.
 * For actions that are attributed to the workflow user account, the global date fields format setting is applied.
 *
 * @param {string} [timeZoneId] The ID of a time zone in which the dateTimeString occurs.
 * This parameter is only effective when the format that matches the string does not provide any timestamp information.
 * If neither the format that successfully matches the string nor this parameter provide the time zone, the time zone is supplied by the system.
 * For actions that are attributed to the current user, the local time zone setting from the profile for the current user is applied.
 * For actions that are attributed to the workflow user account, the global default time zone is applied.
 * For a list of time zone IDs, see {@link http://joda-time.sourceforge.net/timezones.html|JodaTime}.
 *
 * @return {number} A timestamp representation of the specified string.
 */
exports.parse = function (dateTimeString, formats, timeZoneId) {
  if (formats && !Array.isArray(formats)) {
    formats = [formats];
  }
  return utils.toUnixTimestamp(dateTimeString, formats || null, timeZoneId || null);
};

/**
 * Creates a string representation of a Unix timestamp.
 *
 * @param {number} timestamp The timestamp to format as a string.
 *
 * @param {string} [format] The date format to apply to the output.
 * If no value is specified, the date format is supplied by the system.
 * For actions that are attributed to the current user, the date format setting from the profile for the current user is applied.
 * For actions that are attributed to the workflow user account, the global date fields format setting is applied.
 *
 * @param {string} [timeZoneId] The ID of a time zone.
 * Applies an offset to the original timestamp, which is in UTC.
 * If no value is specified, the time zone is supplied by the system.
 * For actions that are attributed to the current user, the local time zone setting from the profile for the current user is applied.
 * For actions that are attributed to the workflow user account, the global default time zone is applied.
 * For a list of time zone IDs, see {@link http://joda-time.sourceforge.net/timezones.html|JodaTime}.
 *
 * @return {string} A string representation of the specified timestamp.
 */
exports.format = function (timestamp, format, timeZoneId) {
  return utils.formatTimestamp(timestamp, format || null, timeZoneId || null);
};

/**
 * Creates a period representation of an argument.
 *
 * @example
 * issue.fields.Estimation = dateTime.toPeriod(3 * 3600 * 1000); // 3h in ms
 * issue.fields.Estimation = dateTime.toPeriod('3h'); // short form
 * issue.fields.Estimation = dateTime.toPeriod('2w4d3h15m'); // full form
 *
 * @param {(number|string)} period A duration in milliseconds as either a number or a string. The string representation is a series of numeric values followed by the abbreviation that represents the timespan, in descending order. For example, 1m3w4d23h58m.
 * @return {Period} The period representation of the specified argument.
 * @see For possible usages, see {@link PeriodProjectCustomField}.
 */
exports.toPeriod = function (period) {
  if (typeof period === 'number') {
    return new Period(period);
  }
  var n = parseInt(period);
  // check that entire 'period' string was an integer presentation
  if (period.toString() === n.toString()) {
    return new Period(n);
  }
  return Period.parse(period, formatter);
};

var amend = function(operation, timestamp, duration) {
  if (!timestamp || !duration) {
    return timestamp;
  }
  if (typeof timestamp !== 'number') {
    throw "Illegal value " + timestamp + " of 'timestamp' argument. Should be of type number";
  }
  if (typeof duration === 'number' || typeof duration === 'string') {
    duration = exports.toPeriod(duration);
  } else if (!(duration instanceof Period)) {
    throw "Unsupported type of 'duration' argument. Should be either a number, or a string or a Period (returned by toPeriod)"
  }
  var date = new joda.DateTime(timestamp);
  return date[operation](duration).getMillis();
};

/**
 * Returns a timestamp that represents a point in time after the specified period from the specified date.
 * @since 2018.2.42881
 * @param {number} timestamp  The base date value.
 * @param {(number|string|Period)} duration A duration as a number (in milliseconds), string representation, or period as retrieved from a custom field or returned by the toPeriod() function. The string representation is a series of numeric values followed by the abbreviation that represents the timespan, in descending order. For example, 1m3w4d23h58m.
 * @return {number} The resulting timestamp.
 */
exports.after = function(timestamp, duration) {
  return amend('plus', timestamp, duration);
};

/**
 * Returns a timestamp that represents a point in time before the specified period from the specified date.
 * @since 2018.2.42881
 * @param {number} timestamp The base date value.
 * @param {(number|string|Period)} duration A duration as a number (in milliseconds), string representation, or period as retrieved from a custom field or returned by the toPeriod() function. The string representation is a series of numeric values followed by the abbreviation that represents the timespan, in descending order. For example, 1m3w4d23h58m.
 * @return {number} The resulting timestamp.
 */
exports.before = function(timestamp, duration) {
  return amend('minus', timestamp, duration);
};