/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

/**
 * Contains definitions for the entities in YouTrack that are accessible to workflows.
 * @module @jetbrains/youtrack-scripting-api/entities
 */

/**
 @typedef {Object} Requirement
 @classdesc
 A single element in a set of {@link Requirements}
 @property {string} [name] The optional name of the field or entity. If not provided,
 the key (alias) for this requirement in the {@link Requirements} object is used.
 @property {string} [login] An optional login, used instead of name for {@link User} requirements.
 @property {string} [id] An optional issue ID, used instead of name for {@link Issue} requirements.
 @property {boolean} [multi] An optional flag, `false` by default. If `true`, a required field
 must store multiple values (if applicable).
 @property {string} [outward] The outward name of the issue link type (required for {@link IssueLinkPrototype} requirements).
 @property {string} [inward] The inward name of the issue link type (equals outward name if not set).
 @property {string|Object} [type] The data type of the entity. Can be one of the following custom field types: {@link Build}.fieldType,
 {@link OwnedField}.fieldType, {@link State}.fieldType, {@link EnumField}.fieldType,
 {@link ProjectVersion}.fieldType, {@link User}.fieldType, {@link UserGroup}.fieldType,
 {@link Field}.dateType, {@link Field}.floatType, {@link Field}.integerType,
 {@link Field}.stringType, {@link Field}.periodType.
 Can also be one of the following system-wide entities: {@link User}, {@link UserGroup},
 {@link Project}, {@link Issue}, {@link IssueTag}, {@link SavedQuery}, {@link IssueLinkPrototype}.
 */

/**
 @typedef {Object<string, Requirement>} Requirements

 @classdesc
 The `Requirements` object serves two purposes.
 First, it functions as a safety net. It specifies the set of entities that must exist for a rule to work as expected.
 Whenever one or more rule requirements are not met, corresponding errors are shown in the workflow administration UI.
 The rule is not executed until all of the problems are fixed.

 Second, it functions as a reference.
 Each entity in the requirements is plugged into the `context` object, so you can reference entities from inside your
 context-dependent functions (like an `action` function).

 There are two types of requirements: project-wide and system-wide.
 Project-wide requirements contain a list of custom fields that must be attached
 to each project that uses the rule as well as the required values from the sets of values for each custom field.
 System-wide requirements contain a list of other entities that must be available in YouTrack.
 This includes users, groups, projects, issues, tags, saved searches, and issue link types.

 @example
 requirements: {
   P: {
     type: entities.EnumField.fieldType,
     name: 'Priority',
     M: { name: 'Major' },
     Normal: {}
   },

   ImportantPerson: {
     type: entities.User,
     login: 'superadmin'
   },
   OurTeam: {
     type: entities.UserGroup,
     name: 'integration-team'
   },
   Int: {
     type: entities.Project,
     name: 'Integration'
   },
   Ref: {
     type: entities.Issue,
     id: 'INT-483'
   },
   ToBeReleased: {
     type: entities.IssueTag,
     name: 'To be released'
   },
   Untested: {
     type: entities.SavedQuery,
     name: 'Not tested yet'
   },
   Depend: {
     type: entities.IssueLinkPrototype,
     outward: 'is required for',
     inward: 'depends on'
   }
 }
 */

/**
 * This function is called by different events depending on the rule type:
 * when a change is applied to an issue (on-change rules), when a command is executed (action rules),
 * or according to a predefined schedule (scheduled rules). This function is called separately
 * for each related issue.
 *
 * @example
 * action: function(ctx) {
 *   console.log('Action for issue ' + ctx.issue.id);
 * }
 *
 * @callback Issue~actionFunction
 *
 * @param {Object} ctx The execution context. Along with the parameters listed below,
 * the context also contains objects that you describe in the {@link Requirements}.
 * @param {Issue} ctx.issue The current issue.
 * @param {User} ctx.currentUser The user who executes the rule.
 * @param {Boolean} ctx.isSilentOperation When `true`, the rule is triggered in reaction to a command that is applied without notification.
 * When `false`, the rule is triggered in reaction to a command that is applied normally. Available since 2017.4.38771.
 */

/**
 * This function is called to determine whether an action function can be applied to an issue.
 * Guard functions are used in on-change rules, action rules, and in transitions between values of a state-machine rule.
 * On-schedule rules also support guard functions, but this rule type includes a `search` property that has a similar purpose.
 *
 * @example
 * guard: function(ctx) {
 *   return ctx.issue.State.isResolved;
 * }
 *
 * @callback Issue~guardFunction
 *
 * @param {Object} ctx The execution context. Along with the parameters listed below,
 * the context also contains objects that you describe as {@link Requirements}.
 * @param {Issue} ctx.issue The current issue.
 * @param {User} ctx.currentUser The user who executes the rule.
 */


/**
 * @typedef {Object} Iterator
 *
 * @classdesc
 * An object that enables traversal through the elements in a collection.
 */
/**
 * @method
 * @name next
 * @memberOf Iterator
 * @return {Object} An object that contains values for the properties `done` and `value` properties.
 * If there are elements that were not traversed, `done` is `false` and `value` is the next element in the collection.
 * If all of the elements were traversed, `done` is `true` and `value` is `null`.
 */

/**
 * @typedef {Object} Set
 *
 * @classdesc
 * The `Set` object stores unique values of any type, whether primitive values or
 * object references. The Set is used as storage for all multi-value objects in
 * this API: custom fields that store multiple values, issue links, issues in a project, and so on.
 * You can access single values in the collection directly (see first(), last(), get(index)),
 * use an iterator (see entries(), values()), or traverse with forEach(visitor)
 * and find(predicate) methods.
 *
 * The workflow API is based on ECMAScript 5.1.
 * This Set implementation mimics the functionality supported by the
 * {@link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Set|ES 6 Set interface}.
 *
 * @property {number} [size] The number of elements in a Set.
 * Use thoughtfully, as the calculation for large collections (like `project.issues`) may be resource consumptive.
 *
 * @property {Set} [added] The elements that are added to a field that stores multiple values in the current transaction.
 * Only relevant when the Set represents a multi-valued property (field) of a persistent entity.
 *
 * @property {Set} [removed] The elements that are removed from a field that stores multiple values in the current transaction.
 * Only relevant when the Set represents a multi-valued property (field) of a persistent entity.
 *
 * @property {boolean} [isChanged] When the Set represents a multi-valued property (field) of a persistent entity
 * and the field is changed in the current transaction, this property is `true`.
 *
 * @returns {boolean} If the field is changed in the current transaction, returns `true`.
 *
 * @returns {Set} The values that were added to the field.
 */
/**
 * Find an element with a specific index position in a Set.
 *
 * @method
 * @instance
 * @name get
 * @memberOf Set
 * @param {number} index The ordinal index of the element to be returned from the Set.
 * @returns {Object} An object at index position in a Set, or null
 * if the Set contains fewer than (index + 1) elements.
 */
/**
 * Find the first object in a collection based on the order in which the elements were added to the Set.
 *
 * @method
 * @instance
 * @name first
 * @memberOf Set
 * @returns {Object} The first object in a collection or null if the collection is empty.
 */
/**
 * Find the last object in a collection based on the order in which the elements were added to the Set.
 *
 * @method
 * @instance
 * @name last
 * @memberOf Set
 * @returns {Object} The last object in a collection or null if collection is empty.
 */
/**
 * Get an iterator for the entries in a Set. The same as `values()`.
 * Use an iterator when you need to traverse over entries until a specific condition
 * is met and modify the entries at the same time.
 *
 * @see If you need to modify all of the elements in the Set, see forEach(visitor).
 * @see If you need to find an element that meets specific criteria, see find(predicate).
 *
 * @example
 * // We want to find first Critical among issue subtasks and assign it.
 * var checkAndAssign = function(task) {
 *   if (task.fields.Priority.name === ctx.Priority.Critical.name) {
 *     task.fields.Assignee = ctx.currentUser;
 *     return true;
 *   }
 *   return false;
 * };
 * var iter = issue.links['parent for'].entries();
 * var v = iter.next();
 * while (!v.done && !checkAndAssign(v.value)) {
 *   v = iter.next();
 * }
 *
 * @method
 * @instance
 * @name entries
 * @memberOf Set
 * @returns {Iterator<Object>} An iterator for the collection of values.
 */
/**
 * Get an iterator for the entries in a Set. The same as `entries()`.
 * @see For details, see entries().
 *
 * @method
 * @instance
 * @name values
 * @memberOf Set
 * @returns {Iterator<Object>} An iterator for the collection of values.
 */
/**
 * Apply a visitor function to each member of a collection.
 *
 * @example
 * issue.links['parent for'].forEach(function(child) {
 *   child.fields.Priority = issue.fields.Priority;
 * });
 *
 * @method
 * @instance
 * @name forEach
 * @memberOf Set
 * @param {function} visitor The function to be applied to each member of the collection.
 */
/**
 * Find the first element in a Set for which a predicate function returns `true`.
 *
 * @example
 * issue.tags.find(function(tag) {
 *   return tag.name === 'release';
 * });
 *
 * @method
 * @instance
 * @name find
 * @memberOf Set
 * @param {function} predicate A function with one argument that returns
 * either true or false for a given value in the Set.
 * @returns {Object} The first value that returns `true` for the predicate function or null.
 */
/**
 * Checks a Set object to determine whether the specified element is present in the collection or not.
 *
 * @method
 * @instance
 * @name has
 * @memberOf Set
 * @param {Object} element The element to locate in the Set.
 * @returns {boolean} If the element is found, returns `true`, otherwise, `false`.
 */
/**
 * Checks a Set object to determine whether it is empty.
 *
 * @method
 * @instance
 * @name isEmpty
 * @memberOf Set
 * @returns {boolean} If the Set is empty, returns `true`, otherwise, `false`.
 */
/**
 * Checks a Set object to determine whether it is not empty.
 *
 * @method
 * @instance
 * @name isNotEmpty
 * @memberOf Set
 * @returns {boolean} If the Set is not empty, returns `true`, otherwise, `false`.
 */
/**
 * Add an element to a Set. If the specified value is already present, a duplicate value is not added.
 * @method
 * @instance
 * @name add
 * @memberOf Set
 * @param {Object} element The element to add to the Set.
 */
/**
 * Remove an element from a Set. If the specified element is not present, nothing happens.
 * @method
 * @instance
 * @name delete
 * @memberOf Set
 * @param {Object} element The element to remove from the Set.
 */
/**
 * Remove all of the values from a Set.
 * @method
 * @instance
 * @name clear
 * @memberOf Set
 */

/**
 * @classdesc The common ancestor for all entity types.
 */
var BaseEntity = function () {
  /**
   * When `true`, the entity is removed in the current transaction. Otherwise, `false`.
   * @name BaseEntity#becomesRemoved
   * @type {Boolean}
   * @readonly
   * @since 2017.4.37915
   */
  this.becomesRemoved = true;

  /**
   * When `true`, the entity is created in the current transaction. Otherwise, `false`.
   * @name BaseEntity#isNew
   * @type {Boolean}
   * @readonly
   * @since 2018.2.42351
   */
  this.isNew = true;
};

/**
 * Asserts that a value is set for a field.
 * If a value for the required field is not set, the specified message is displayed in the user interface.
 *
 * @param {string} fieldName The name of the field to check.
 * @param {string} message The message that is displayed to the user that describes the field requirement.
 */
BaseEntity.prototype.required = function (fieldName, message) {
};

/**
 * Checks whether a field is set to an expected value in the current transaction.
 *
 * @param {string} fieldName The name of the field to check.
 * @param {string} expected The expected value.
 *
 * @returns {boolean} If the field is set to the expected value, returns `true`.
 */
BaseEntity.prototype.becomes = function (fieldName, expected) {
};

/**
 * Checks whether the value of a field is changed in the current transaction.
 *
 * @param {string} fieldName The name of the field to check.
 *
 * @returns {boolean} If the value of the field is changed in the current transaction, returns `true`.
 */
BaseEntity.prototype.isChanged = function (fieldName) {
};

/**
 * Returns the previous value of a single-value field before an update was applied. If the field is not changed
 * in the transaction, returns null.
 *
 * @param {string} fieldName The name of the field.
 *
 * @returns {Object} If the field is changed in the current transaction, the previous value of the field.
 * Otherwise, null.
 */
BaseEntity.prototype.oldValue = function (fieldName) {
};

/**
 * Checks whether a user has permission to read the field.
 *
 * @param {string} fieldName The name of the field.
 * @param {User} user The user for whom the permission to read the field is checked.
 *
 * @returns {boolean} If the user can read the field, returns `true`.
 */
BaseEntity.prototype.canBeReadBy = function (fieldName, user) {
};

/**
 * Checks whether a user has permission to update the field.
 *
 * @param {string} fieldName The name of the field.
 * @param {User} user The user for whom the permission to update the field is checked.
 *
 * @returns {boolean} If the user can update the field, returns `true`.
 */
BaseEntity.prototype.canBeWrittenBy = function (fieldName, user) {
};



/**
 * @classdesc Represents an agile board and the set of sprints that belong to the board.
 * @augments BaseEntity
 */
var Agile = function () {
  /**
   * The user who created the board.
   * @name Agile#author
   * @type {User}
   * @readonly
   */
  this.author = {/*this code is hidden*/};
  /**
   * The name of the agile board.
   * @name Agile#name
   * @type {String}
   * @readonly
   */
  this.name = '';
  /**
   * The set of sprints that are associated with the board.
   * @name Agile#sprints
   * @type {Set<Sprint>}
   * @readonly
   */
  this.sprints = {/*this code is hidden*/};
};
Agile.prototype = Object.create(BaseEntity.prototype);
Agile.prototype.constructor = BaseEntity;

/**
 * Finds a specific sprint by name.
 * @param {String} name The name of the sprint.
 *
 * @returns {Sprint} If a sprint with the specified name is found, the corresponding Sprint object is returned. Otherwise, the return value is null.
 */
Agile.prototype.findSprintByName = function (name) {
  return {/*this code is hidden*/};
};

/**
 * Returns the sprints that an issue is assigned to on an agile board.
 * @since 2018.1.39547
 * @param {Issue} issue The issue for which you want to get the sprints that it is assigned to.
 *
 * @returns {Set<Sprint>} The sprints that the issue is assigned to on the agile board.
 */
Agile.prototype.getIssueSprints = function (issue) {
  return {/*this code is hidden*/};
};

/**
 * Returns a set of agile boards that have the specified name.
 * @static
 * @param {String} name The name of an agile board.
 *
 * @returns {Set<Agile>} A set of agile boards that are assigned the specified name.
 */
Agile.findByName = function (name) {
  return {/*this code is hidden*/};
};


/**
 * @classdesc The base class for issue comment.
 * @augments BaseEntity
 */
var BaseComment = function () {
  /**
   * The set of attachments that are attached to the comment.
   * @name BaseComment#attachments
   * @type {Set<IssueAttachment>}
   * @readonly
   * @since 2018.1.40030
   */
  this.attachments = {/*this code is hidden*/};
  /**
   * When `true`, the comment text is parsed as Markdown. When `false`, the comment text is parsed as YouTrack Wiki.
   * Changing this value does not transform the markup from one syntax to another.
   * @name BaseComment#isUsingMarkdown
   * @type {Boolean}
   * @since 2017.4.38870
   */
  this.isUsingMarkdown = true;
};
BaseComment.prototype = Object.create(BaseEntity.prototype);
BaseComment.prototype.constructor = BaseEntity;


/**
 * @classdesc The base class for issue work items.
 * @augments BaseEntity
 */
var BaseWorkItem = function () {
  /**
   * When `true`, the work item description is parsed as Markdown. When `false`, the work item description is parsed as YouTrack Wiki.
   * Changing this value does not transform the markup from one syntax to another.
   * @name BaseWorkItem#isUsingMarkdown
   * @type {Boolean}
   * @since 2017.4.38870
   */
  this.isUsingMarkdown = true;
  /**
   * The user to whom the work is attributed in the work item.
   * @name BaseWorkItem#author
   * @type {User}
   * @readonly
   */
  this.author = {/*this code is hidden*/};
  /**
   * The date when the work item was created.
   * @name BaseWorkItem#created
   * @type {Number}
   * @readonly
   */
  this.created = 0;
  /**
   * The user who added the work item to the issue.
   * @name BaseWorkItem#creator
   * @type {User}
   * @readonly
   */
  this.creator = {/*this code is hidden*/};
  /**
   * The work item description.
   * @name BaseWorkItem#description
   * @type {String}
   */
  this.description = '';
  /**
   * The work item type.
   * @name BaseWorkItem#type
   * @type {WorkItemType}
   * @readonly
   */
  this.type = {/*this code is hidden*/};
  /**
   * The date when the work item was last updated.
   * @name BaseWorkItem#updated
   * @type {Number}
   * @readonly
   */
  this.updated = 0;
};
BaseWorkItem.prototype = Object.create(BaseEntity.prototype);
BaseWorkItem.prototype.constructor = BaseEntity;


/**
 * @classdesc Represents a value that is stored in a custom field that stores a build type.
 * @augments Field
 */
var Build = function () {
  /**
   * Field type. Used when defining rule requirements.
   * @name Build.fieldType
   * @type {String}
   * @readonly
   */
  this.fieldType = '';
  /**
   * The date and time when the build was assembled.
   * @name Build#assembleDate
   * @type {Number}
   * @readonly
   */
  this.assembleDate = 0;
};
Build.prototype = Object.create(Field.prototype);
Build.prototype.constructor = Field;


/**
 * @classdesc Represents a custom field in a project that stores an predefined set of values.
 * @augments ProjectCustomField
 */
var BundleProjectCustomField = function () {
  /**
   * The set of values that is available for the custom field.
   * @name BundleProjectCustomField#values
   * @type {Set<Field>}
   * @readonly
   */
  this.values = {/*this code is hidden*/};
};
BundleProjectCustomField.prototype = Object.create(ProjectCustomField.prototype);
BundleProjectCustomField.prototype.constructor = ProjectCustomField;

/**
 * Adds a value to the set of values for the custom field. If a value with the specified name already exists in the set, an exception is thrown.
 * @since 2018.2.45017
 * @param {String} name The name of the value that you want to add to the set.
 *
 * @returns {Field} The value that was added to the set.
 */
BundleProjectCustomField.prototype.createValue = function (name) {
  return {/*this code is hidden*/};
};

/**
 * Returns a value with the specified name in the set of values for a custom field.
 * @param {String} name The name of the field value to search for.
 *
 * @returns {Field} The value with the specified name in the set of values for the custom field.
 */
BundleProjectCustomField.prototype.findValueByName = function (name) {
  return {/*this code is hidden*/};
};

/**
 * Returns a value that is assigned a specified position in the set of values for a custom field.
 * @param {Number} ordinal The position of the field value to search by.
 *
 * @returns {Field} The value that is assigned the specified position in the set of values for the custom field.
 */
BundleProjectCustomField.prototype.findValueByOrdinal = function (ordinal) {
  return {/*this code is hidden*/};
};


/**
 * @classdesc An entity that retrieves VCS changes and creates their representation in YouTrack.
 * @augments BaseEntity
 */
var ChangesProcessor = function () {
  /**
   * The VCS server that the processor connects to.
   * @name ChangesProcessor#server
   * @type {VcsServer}
   * @readonly
   * @since 2018.1.38923
   */
  this.server = {/*this code is hidden*/};
};
ChangesProcessor.prototype = Object.create(BaseEntity.prototype);
ChangesProcessor.prototype.constructor = BaseEntity;


/**
 * @classdesc Represents a value in a custom field that stores a predefined set of values.
 * @augments Field
 */
var EnumField = function () {
  /**
   * Field type. Used when defining rule requirements.
   * @name EnumField.fieldType
   * @type {String}
   * @readonly
   */
  this.fieldType = '';
};
EnumField.prototype = Object.create(Field.prototype);
EnumField.prototype.constructor = Field;


/**
 * @classdesc Represents a value that is stored in a custom field.
 * The custom fields themselves are represented by the Fields class.
 * @augments BaseEntity
 */
var Field = function () {
  /**
   * The background color of the value in the custom field as it is displayed in YouTrack.
   * @name Field#backgroundColor
   * @type {String}
   * @readonly
   */
  this.backgroundColor = '';
  /**
   * The foreground color of the value in the custom field as it is displayed in YouTrack.
   * @name Field#foregroundColor
   * @type {String}
   * @readonly
   */
  this.foregroundColor = '';
  /**
   * String representation of the value.
   * @name Field#presentation
   * @type {String}
   * @readonly
   */
  this.presentation = '';
  /**
   * Date and time field type. Used when defining rule requirements.
   * @name Field.dateTimeType
   * @type {String}
   * @readonly
   */
  this.dateTimeType = '';
  /**
   * Date field type. Used when defining rule requirements.
   * @name Field.dateType
   * @type {String}
   * @readonly
   */
  this.dateType = '';
  /**
   * Float field type. Used when defining rule requirements.
   * @name Field.floatType
   * @type {String}
   * @readonly
   */
  this.floatType = '';
  /**
   * Integer field type. Used when defining rule requirements.
   * @name Field.integerType
   * @type {String}
   * @readonly
   */
  this.integerType = '';
  /**
   * Period field type. Used when defining rule requirements.
   * @name Field.periodType
   * @type {String}
   * @readonly
   */
  this.periodType = '';
  /**
   * String field type. Used when defining rule requirements.
   * @name Field.stringType
   * @type {String}
   * @readonly
   */
  this.stringType = '';
  /**
   * Text field type. Used when defining rule requirements.
   * @name Field.textType
   * @type {String}
   * @readonly
   */
  this.textType = '';
  /**
   * The index value of the color that is assigned to the value in the custom field.
   * @name Field#colorIndex
   * @type {Number}
   * @readonly
   */
  this.colorIndex = 0;
  /**
   * The description of the value as visible in the administrative UI for custom fields.
   * @name Field#description
   * @type {String}
   * @readonly
   */
  this.description = '';
  /**
   * The name of the value, which is also stored as the value in the custom field.
   * @name Field#name
   * @type {String}
   * @readonly
   */
  this.name = '';
  /**
   * The position of the value in the set of values for the custom field.
   * @name Field#ordinal
   * @type {Number}
   * @readonly
   */
  this.ordinal = 0;
};
Field.prototype = Object.create(BaseEntity.prototype);
Field.prototype.constructor = BaseEntity;


/**
 * @classdesc Represents a custom field in a project that stores a UserGroup type.
 * @augments ProjectCustomField
 */
var GroupProjectCustomField = function () {
  /**
   * The set of available values for this custom field.
   * @name GroupProjectCustomField#values
   * @type {Set<UserGroup>}
   * @readonly
   */
  this.values = {/*this code is hidden*/};
  /**
   * The set of default values.
   * @name GroupProjectCustomField#defaultValues
   * @type {Set<UserGroup>}
   */
  this.defaultValues = {/*this code is hidden*/};
};
GroupProjectCustomField.prototype = Object.create(ProjectCustomField.prototype);
GroupProjectCustomField.prototype.constructor = ProjectCustomField;

/**
 * Returns the value that matches the specified name in a custom field that stores values as a user group.
 * @param {String} name The name of the group to search for in the set of values for the custom field.
 *
 * @returns {UserGroup} The group with the specified name. This group can be set as the value for a field that stores a user group.
 */
GroupProjectCustomField.prototype.findValueByName = function (name) {
  return {/*this code is hidden*/};
};


/**
 * @typedef {object} JsonForIssueConstructor
 * @property {User} reporter Issue reporter.
 * @property {Project} project Project that the new issue is to belong to.
 * @property {String} summary Issue summary.
 */
/**
 * @classdesc Represents an issue in YouTrack.
 * @augments BaseEntity
 *
 * @constructor Issue
 * @param {User|JsonForIssueConstructor} reporter Issue reporter. Alternatively, pass a JSON specified by {@link JsonForIssueConstructor}
 * @param {Project} [project] Project that the new issue is to belong to.
 * @param {String} [summary] Issue summary.
 */
var Issue = function (reporter, project, summary) {
  /**
   * If the issue becomes reported in the current transaction, this property is `true`.
   * @example
   * if (issue.fields.Subsystem !== null && issue.fields.Assignee === null &&
   *     (((issue.isChanged('Subsystem') || issue.isChanged('project') && issue.isReported) ||
   *     issue.becomesReported) {
   *   issue.fields.Assignee = issue.fields.Subsystem.owner
   * }
   * @name Issue#becomesReported
   * @type {Boolean}
   * @readonly
   */
  this.becomesReported = true;
  /**
   * If the issue is assigned a state that is considered resolved in the current transaction, this property is `true`.
   * @name Issue#becomesResolved
   * @type {Boolean}
   * @readonly
   */
  this.becomesResolved = true;
  /**
   * If the issue is assigned a state that is considered unresolved in the current transaction. this property is `true`.
   * @name Issue#becomesUnresolved
   * @type {Boolean}
   * @readonly
   */
  this.becomesUnresolved = true;
  /**
   * The root issue in a tree of duplicates that are linked to the issue.
   * For example, if `issueA` duplicates `issueB` and `issueB` duplicates
   * `issueC`, then the value for the `issueA.duplicateRoot()` property is `issueC`.
   * @name Issue#duplicateRoot
   * @type {Issue}
   * @readonly
   */
  this.duplicateRoot = {/*this code is hidden*/};
  /**
   * The set of comments that are edited in the current transaction.
   * Comments that are added and removed are not considered to be edited.
   * Instead, these are represented by the `issue.comments.added` and
   * `issue.comments.removed` properties.
   * @name Issue#editedComments
   * @type {Set<IssueComment>}
   * @readonly
   */
  this.editedComments = {/*this code is hidden*/};
  /**
   * The set of work items that are edited in the current transaction.
   * Work items that are added and removed are not considered to be edited.
   * Instead, these are represented by the `issue.workItems.added` and
   * `issue.workItems.removed` properties.
   * @name Issue#editedWorkItems
   * @type {Set<IssueWorkItem>}
   * @readonly
   * @since 2017.4.37824
   */
  this.editedWorkItems = {/*this code is hidden*/};
  /**
   * The visible ID of an issue or similar object in an originating third-party system.
   * @name Issue#externalId
   * @type {String}
   * @readonly
   * @since 2018.2.42312
   */
  this.externalId = '';
  /**
   * The URL for an issue or similar object in an originating third-party system.
   * @name Issue#externalUrl
   * @type {String}
   * @readonly
   * @since 2018.2.42312
   */
  this.externalUrl = '';
  /**
   * The issue ID.
   * @example
   * user.notify('Issue is overdue', 'Please, look at the issue: ' + issue.id);
   * @name Issue#id
   * @type {String}
   * @readonly
   */
  this.id = '';
  /**
   * The absolute URL that points to the issue.
   * @example
   * user.notify('Issue is overdue', 'Please, look at the issue: ' + issue.url);
   * @name Issue#url
   * @type {String}
   * @readonly
   */
  this.url = '';
  /**
   * The list of VCS changes that are associated with the issue.
   * @name Issue#vcsChanges
   * @type {Set<VcsChange>}
   * @readonly
   * @since 2018.1.38923
   */
  this.vcsChanges = {/*this code is hidden*/};
  /**
   * The set of work items that have been added to the issue.
   * @name Issue#workItems
   * @type {Set<IssueWorkItem>}
   * @readonly
   */
  this.workItems = {/*this code is hidden*/};
  /**
   * If the issue is already reported or becomes reported in the current transaction, this property is `true`. To apply changes to an issue draft, use `!issue.isReported`.
   * @example
   * issue.links['depends on'].forEach(function(dep) {
   *   if (dep.isReported) {
   *     assert(dep.State.resolved, 'The issue has unresolved dependencies and thus cannot be set Fixed!');
   *   }
   * });
   * @name Issue#isReported
   * @type {Boolean}
   * @readonly
   */
  this.isReported = true;
  /**
   * If the issue is currently assigned a state that is considered resolved, this property is `true`.
   * @name Issue#isResolved
   * @type {Boolean}
   * @readonly
   */
  this.isResolved = true;
  /**
   * If the current user has added the star tag to watch the issue, this property is `true`.
   * @name Issue#isStarred
   * @type {Boolean}
   * @readonly
   */
  this.isStarred = true;
  /**
   * The set of attachments that are attached to the issue.
   * @name Issue#attachments
   * @type {Set<IssueAttachment>}
   * @readonly
   */
  this.attachments = {/*this code is hidden*/};
  /**
   * A list of comments for the issue.
   * @name Issue#comments
   * @type {Set<IssueComment>}
   * @readonly
   */
  this.comments = {/*this code is hidden*/};
  /**
   * The date when the issue was created.
   * @name Issue#created
   * @type {Number}
   * @readonly
   */
  this.created = 0;
  /**
   * The text that is entered as the issue description.
   * @name Issue#description
   * @type {String}
   */
  this.description = '';
  /**
   * When `true`, the issue description is parsed as Markdown. When `false`, the issue description is parsed as YouTrack Wiki.
   * Changing this value does not transform the markup from one syntax to another.
   * @name Issue#isUsingMarkdown
   * @type {Boolean}
   * @since 2017.4.38870
   */
  this.isUsingMarkdown = true;
  /**
   * The issue number in the project.
   * @name Issue#numberInProject
   * @type {Number}
   * @readonly
   */
  this.numberInProject = 0;
  /**
   * The user group for which the issue is visible. If the property contains a null value, the issue is visible to the All Users group.
   * @name Issue#permittedGroup
   * @type {UserGroup}
   */
  this.permittedGroup = {/*this code is hidden*/};
  /**
   * The groups for which the issue is visible when the visibility is restricted to multiple groups.
   * @name Issue#permittedGroups
   * @type {Set<UserGroup>}
   */
  this.permittedGroups = {/*this code is hidden*/};
  /**
   * The list of users for whom the issue is visible.
   * @name Issue#permittedUsers
   * @type {Set<User>}
   */
  this.permittedUsers = {/*this code is hidden*/};
  /**
   * The project to which the issue is assigned.
   * @name Issue#project
   * @type {Project}
   */
  this.project = {/*this code is hidden*/};
  /**
   * The user who reported (created) the issue.
   * @example
   * issue.fields.Assignee = issue.reporter;
   * @name Issue#reporter
   * @type {User}
   * @readonly
   */
  this.reporter = {/*this code is hidden*/};
  /**
   * The date and time when the issue was assigned a state that is considered to be resolved.
   * @name Issue#resolved
   * @type {Number}
   * @readonly
   */
  this.resolved = 0;
  /**
   * The collection of sprints that this issue has been added to.
   * @name Issue#sprints
   * @type {Set<Sprint>}
   */
  this.sprints = {/*this code is hidden*/};
  /**
   * The text that is entered as the issue summary.
   * @name Issue#summary
   * @type {String}
   */
  this.summary = '';
  /**
   * The list of tags that are attached to an issue.
   * @name Issue#tags
   * @type {Set<IssueTag>}
   */
  this.tags = {/*this code is hidden*/};
  /**
   * The date when the issue was last updated.
   * @name Issue#updated
   * @type {Number}
   * @readonly
   */
  this.updated = 0;
  /**
   * The user who last updated the issue.
   * @name Issue#updatedBy
   * @type {User}
   * @readonly
   */
  this.updatedBy = {/*this code is hidden*/};
  /**
   * The number of votes for an issue. For vote-related methods, see User.canVoteIssue, User.voteIssue, User.canUnvoteIssue, and User.unvoteIssue.
   * @name Issue#votes
   * @type {Number}
   * @readonly
   */
  this.votes = 0;

  /**
   * Asserts that a value is set for a custom field.
   * If a value for the required field is not set, the specified message is displayed in the user interface.
   *
   * @method
   * @name required
   * @memberOf Fields
   *
   * @param {string} fieldName The name of the custom field to check.
   * @param {string} message The message that is displayed to the user that describes the field requirement.
   */
  /**
   * Checks whether the value for a custom field is set to an expected value in the current transaction.
   *
   * @method
   * @name becomes
   * @memberOf Fields
   *
   * @param {string|ProjectCustomField} field The field to check.
   * @param {Object} expected The expected value.
   *
   * @returns {boolean} If the field is set to the expected value, returns `true`.
   */
  /**
   * Checks whether the custom field is changed in the current transaction.
   *
   * @method
   * @name isChanged
   * @memberOf Fields
   *
   * @param {string|ProjectCustomField} field The name of the custom field (for example, 'State') or a reference to the field that is checked.
   *
   * @returns {boolean} If the value of the field is changed in the current transaction, returns `true`.
   */
  /**
   * Returns the previous value of a single-valued custom field before an update was applied. If the field is not changed
   * in the transaction, this value is equal to the current value of the field.
   *
   * @method
   * @name oldValue
   * @memberOf Fields
   *
   * @param {string|ProjectCustomField} field The name of the custom field (for example, 'State') or a reference to the field for which the previous value is returned.
   *
   * @returns {Object} If the custom field is changed in the current transaction, the previous value of the field. Otherwise, the current value of the field.
   */
  /**
   * Checks whether a user has permission to read the custom field.
   *
   * @method
   * @name canBeReadBy
   * @memberOf Fields
   *
   * @param {string|ProjectCustomField} field The custom field to check for read access.
   * @param {User} user The user for whom the permission to read the custom field is checked.
   *
   * @returns {boolean} If the user can read the field, returns `true`.
   */
  /**
   * Checks whether a user has permission to update the custom field.
   *
   * @method
   * @name canBeWrittenBy
   * @memberOf Fields
   *
   * @param {string|ProjectCustomField} field The custom field to check for update access.
   * @param {User} user The user for whom the permission to update the custom field is checked.
   *
   * @returns {boolean} If the user can update the field, returns `true`.
   */
  /**
   * @typedef {Object} Fields
   * @classdesc
   * Represents the custom fields that are used in an issue.
   * The actual set of custom fields that are used for each issue is configured on a per-project basis.
   * The properties shown here correspond with the default custom fields in YouTrack.
   * Additional custom fields that have been attached to a project are also available.
   */
  /**
   * The custom fields that are used in an issue. This is the collection of issue attributes like
   * `Assignee`, `State`, and `Priority` that are defined in the Custom Fields section of the administrative interface and
   * can be attached to each project independently.
   *
   * Issue attributes like `reporter`, `numberInProject`, and `project` are accessed directly.
   *
   * @example
   * if (issue.fields.becomes(ctx.Priority, ctx.Priority.Critical) {
   *   issue.fields.Assignee = issue.project.leader;
   * }
   * @name Issue#fields
   * @type {Fields}
   */
  this.fields = {
    /**
     * A custom field that stores data as a Set<ProjectVersion> type.
     * @type {Set<ProjectVersion>}
     * @name Fields#['Affected versions']
     */
    'Affected versions': null,
    /**
     * A custom field that stores data as a User type.
     * @type {User}
     * @name Fields#Assignee
     */
    'Assignee': null,
    /**
     * A custom field that stores data as a Number type.
     * @type {Number}
     * @name Fields#['Due Date']
     */
    'Due Date': null,
    /**
     * A custom field that stores data as a Set<ProjectVersion> type.
     * @type {Set<ProjectVersion>}
     * @name Fields#['Fix versions']
     */
    'Fix versions': null,
    /**
     * A custom field that stores data as a Build type.
     * @type {Build}
     * @name Fields#['Fixed in build']
     */
    'Fixed in build': null,
    /**
     * A custom field that stores data as a EnumField type.
     * @type {EnumField}
     * @name Fields#Priority
     */
    'Priority': null,
    /**
     * A custom field that stores data as a State type.
     * @type {State}
     * @name Fields#State
     */
    'State': null,
    /**
     * A custom field that stores data as a OwnedField type.
     * @type {OwnedField}
     * @name Fields#Subsystem
     */
    'Subsystem': null,
    /**
     * A custom field that stores data as a EnumField type.
     * @type {EnumField}
     * @name Fields#Type
     */
    'Type': null
  };

  /**
   * Issue links (e.g. `relates to`, `parent for`, etc.). Each link is a
   * {@link Set} of {@link Issue} objects.
   * @example
   * if (issue.links['parent for'].added.isNotEmpty()) {
   *   issue.links['parent for'].added.forEach(function(subtask) {
   *     subtask.fields.Priority = issue.fields.Priority;
   *   });
   * }
   * @name Issue#links
   * @type {Object}
   */
  this.links = {
    'relates to': null,
    'depends on': null,
    'is required for': null,
    'duplicates': null,
    'is duplicated by': null,
    'subtask of': null,
    'parent for': null
  };
};
Issue.prototype = Object.create(BaseEntity.prototype);
Issue.prototype.constructor = BaseEntity;

/**
 * @typedef {object} JsonForIssueAddComment
 * @property {String} text The text to add to the issue as a comment.
 * @property {User} [author] The author of the comment.
 */
/**
 * Adds a comment to the issue.
 * Makes `issue.comments.isChanged` return `true` for the current transaction.
 * @param {String|JsonForIssueAddComment} text The text to add to the issue as a comment. Alternatively, pass a JSON specified by {@link JsonForIssueAddComment}
 * @param {User} [author] The author of the comment.
 *
 * @returns {IssueComment} A newly created comment.
 */
Issue.prototype.addComment = function (text, author) {
  return {/*this code is hidden*/};
};

/**
 * Adds a tag with the specified name to an issue. YouTrack adds the first matching tag that is visible to the current user.
 * If a match is not found, a new private tag is created for the current user.
 * When you use this method to add the star tag to an issue, the current user is added to the list of watchers.
 * To add a tag that is not visible to the current user, use the applyCommand method instead.
 * Use "add tag [tagName]" for the command parameter and specify the login for the owner of the tag in the runAs parameter.
 * @example
 * Issue.addTag('doit');
 * @param {String} name The name of the tag to add to the issue.
 *
 * @returns {IssueTag} The tag that has been added to the issue.
 */
Issue.prototype.addTag = function (name) {
  return {/*this code is hidden*/};
};

/**
 * @typedef {object} JsonForIssueAddWorkItem
 * @property {String} description The description of the work item.
 * @property {Number} [date] The date that is assigned to the work item.
 * @property {User} [author] The user who performed the work.
 * @property {Number} duration The work duration in minutes.
 * @property {WorkItemType} [type] The work item type.
 */
/**
 * Adds a work item to the issue.
 * @param {String|JsonForIssueAddWorkItem} description The description of the work item. Alternatively, pass a JSON specified by {@link JsonForIssueAddWorkItem}
 * @param {Number} [date] The date that is assigned to the work item.
 * @param {User} [author] The user who performed the work.
 * @param {Number} [duration] The work duration in minutes.
 * @param {WorkItemType} [type] The work item type.
 *
 * @returns {IssueWorkItem} The new work item.
 */
Issue.prototype.addWorkItem = function (description, date, author, duration, type) {
  return {/*this code is hidden*/};
};

/**
 * Applies a command to the issue.
 * @param {String} command The command that is applied to the issue.
 * @param {User} runAs Specifies the user by which the command is applied. If this parameter is not set, the command is applied on behalf of the current user.
 */
Issue.prototype.applyCommand = function (command, runAs) {};

/**
 * Removes all of the attachments from the issue.
 */
Issue.prototype.clearAttachments = function () {};

/**
 * Creates a copy of the issue.
 * @param {Project} [project] Project to create new issue in. Available since 2018.1.40575.
 *
 * @returns {Issue} The copy of the original issue.
 */
Issue.prototype.copy = function (project) {
  return {/*this code is hidden*/};
};

/**
 * Checks whether the specified tag is attached to an issue.
 * @param {String} tagName The name of the tag to check for the issue.
 *
 * @returns {Boolean} If the specified tag is attached to the issue, returns `true`.
 */
Issue.prototype.hasTag = function (tagName) {
  return true;
};

/**
 * Checks whether the issue is accessible by specified user.
 * @param {User} user The user to check.
 *
 * @returns {Boolean} If the issue is accessbie for the user, returns 'true'.
 */
Issue.prototype.isVisibleTo = function (user) {
  return true;
};

/**
 * Removes a tag with the specified name from an issue. If the specified tag is not attached to the issue, an exception is thrown.
 * This method first searches through tags owned by the current user, then through all other visible tags.
 * To remove a tag that is not visible to the current user, use the applyCommand method instead.
 * Use "remove tag [tagName]" for the command parameter and specify the login for the owner of the tag in the runAs parameter.
 * @example
 * Issue.removeTag('waiting for reply');
 * @param {String} name The name of the tag to remove from the issue.
 *
 * @returns {IssueTag} The tag that has been removed from the issue.
 */
Issue.prototype.removeTag = function (name) {
  return {/*this code is hidden*/};
};

/**
 * Converts text with Markdown or YouTrack Wiki markup to HTML. Use this method to send "pretty" notifications.
 * @example
 * issue.Assignee.notify('Comment added:', issue.wikify(comment.text, comment.isUsingMarkdown));
 * @param {String} text The string of text to convert to HTML.
 * @param {Boolean} [usingMarkdown] If `true`, the markup is parsed as Markdown. If `false`, the markup is parsed as YouTrack Wiki. If omitted, issue.isUsingMarkdown is used implicitly. Available since 2018.1.40100
 *
 * @returns {String} The wikified string.
 */
Issue.prototype.wikify = function (text, usingMarkdown) {
  return '';
};

/**
 * Creates a declaration of a rule that a user can apply to one or more issues with a command or menu option.
 * The object that is returned by this method is normally exported to the `rule` property, otherwise it is not treated as a rule.
 * @example
 * var entities = require('@jetbrains/youtrack-scripting-api/entities');
 * exports.rule = entities.Issue.action({
 *   title: 'Log comments',
 *   command: 'log',
 *   guard: function(ctx) {
 *     return ctx.issue.isReported;
 *   },
 *   action: function(ctx) {
 *     ctx.issue.comments.forEach(function(comment) {
 *       console.log(comment.text);
 *     });
 *   }
 * });
 * @param {Object} ruleProperties A JSON object that defines the properties for the rule.
 * @param {string} ruleProperties.title The human-readable name of the rule. Displayed in the administrative UI in YouTrack.
 * @param {string} ruleProperties.command The custom command that triggers the action.
 * @param {Issue~guardFunction} ruleProperties.guard A function that is invoked to determine whether the action is applicable to an issue.
 * @param {Issue~actionFunction} ruleProperties.action The function that is invoked when a user triggers this action.
 * @param {Requirements} ruleProperties.requirements The set of entities that must be present for the script to work as expected.
 * @static
 *
 * @returns {Object} The object representation of the rule.
 */
Issue.action = function (ruleProperties) {
  return {/*this code is hidden*/};
};

/**
 * Finds an issue by its visible ID.
 * @static
 * @param {String} id The issue ID.
 *
 * @returns {Issue} The issue that is assigned the specified ID.
 */
Issue.findById = function (id) {
  return {/*this code is hidden*/};
};

/**
 * Creates a declaration of a rule that is triggered when a change is applied to an issue.
 * The object that is returned by this method is normally exported to the `rule` property, otherwise it is not treated as a rule.
 * @example
 * var entities = require('@jetbrains/youtrack-scripting-api/entities');
 * exports.rule = entities.Issue.onChange({
 *   title: 'On issue change, log its id',
 *   action: function(ctx) {
 *     console.log(ctx.issue.id);
 *   }
 * });
 * @param {Object} ruleProperties A JSON object that defines the properties for the rule.
 * @param {string} ruleProperties.title The human-readable name of the rule. Displayed in the administrative UI in YouTrack.
 * @param {Issue~actionFunction} ruleProperties.action The function that is invoked on issue change.
 * @param {Requirements} ruleProperties.requirements The set of entities that must be present for the script to work as expected.
 * @param {Object} [ruleProperties.runOn] Determines which issue events trigger the on-change rule. When not specified, the rule is triggered on issue change.
 * @param {boolean} [ruleProperties.runOn.change] When `true`, the rule is triggered on issue change.
 * @param {boolean} [ruleProperties.runOn.removal] When `true`, the rule is triggered when an issue is logically deleted.
 * @static
 *
 * @returns {Object} The object representation of the rule.
 */
Issue.onChange = function (ruleProperties) {
  return {/*this code is hidden*/};
};

/**
 * Creates a declaration of a rule that is triggered on a set schedule.
 * The object that is returned by this method is normally exported to the `rule` property, otherwise it is not treated as a rule.
 * @example
 * var entities = require('@jetbrains/youtrack-scripting-api/entities');
 * exports.rule = entities.Issue.onSchedule({
 *   title: 'Log id of major issues every 5 seconds',
 *   search: '#Major',
 *   cron: '0/5 * * * * ?',
 *   action: function(ctx) {
 *     console.log(ctx.issue.id);
 *   }
 * });
 * @param {Object} ruleProperties A JSON object that defines the properties for the rule.
 * @param {string} [ruleProperties.title] The human-readable name of the rule. Displayed in the administrative UI in YouTrack.
 * @param {string|function} ruleProperties.search A YouTrack search string or a function with no parameters that returns such a string.
 * The specified action is applied to all issues that match the search and belong to the project that this rule is attached to.
 * @param {string} ruleProperties.cron A cron expression that specifies the interval for applying the rule.
 * @param {boolean} [ruleProperties.muteUpdateNotifications] `true` if no notifications should be sent on changes made by this rule or any rule that reacted on a change made by this rule.
 * @param {Issue~actionFunction} ruleProperties.action The function that is invoked on schedule for each issue that matches the search.
 * @param {Requirements} ruleProperties.requirements The set of entities that must be present for the script to work as expected.
 * @static
 *
 * @returns {Object} The object representation of the rule.
 */
Issue.onSchedule = function (ruleProperties) {
  return {/*this code is hidden*/};
};

/**
 * Creates declaration of a state-machine rule. The state-machine imposes restrictions for the transitions between values in a custom field.
 * You can execute actions when the custom field is set to a value, changes from a value, or transitions from two specific values.
 * The object that is returned by this method is normally exported to the `rule` property, otherwise it is not treated as a rule.
 * @example
 * var entities = require('@jetbrains/youtrack-scripting-api/entities');
 * exports.rule = entities.Issue.stateMachine({
 *   title: 'Status state-machine',
 *   fieldName: 'Status',
 *   states: {
 *     Open: {
 *       initial: true,
 *       transitions: {
 *         start: {
 *           targetState: 'In progress'
 *         }
 *       }
 *     },
 *     'In progress': {
 *       onEnter: function(ctx) {
 *         ctx.issue.fields.Assignee = ctx.currentUser;
 *       },
 *       transitions: {
 *         fix: {
 *           targetState: 'Fixed'
 *         },
 *         reopen: {
 *           targetState: 'Open'
 *         }
 *       }
 *     },
 *     Fixed: {
 *       transitions: {
 *       }
 *     }
 *   },
 *   requirements: {
 *     Assignee: {
 *       type: entities.User.fieldType
 *     }
 *   }
 * });
 * @param {Object} ruleProperties A JSON object that defines the properties for the rule.
 * @param {string} ruleProperties.title A human-readable name of the rule. Displayed in the administrative UI in YouTrack.
 * @param {Object[]} ruleProperties.states A list of values for a custom field and the possible transitions between them.
 * @param {Requirements} ruleProperties.requirements The set of entities that must be present for the script to work as expected.
 * @static
 *
 * @returns {Object} The object representation of the rule.
 */
Issue.stateMachine = function (ruleProperties) {
  return {/*this code is hidden*/};
};


/**
 * @classdesc Represents a file that is attached to an issue.
 * @augments PersistentFile
 */
var IssueAttachment = function () {
  /**
   * The user who attached the file to the issue.
   * @name IssueAttachment#author
   * @type {User}
   * @readonly
   */
  this.author = {/*this code is hidden*/};
  /**
   * The date and time when the attachment was created as a timestamp.
   * @name IssueAttachment#created
   * @type {Number}
   * @readonly
   */
  this.created = 0;
  /**
   * If the attachment is removed, this property is `true`.
   * @name IssueAttachment#isRemoved
   * @type {Boolean}
   * @readonly
   */
  this.isRemoved = true;
  /**
   * The issue that the file is attached to.
   * @name IssueAttachment#issue
   * @type {Issue}
   * @readonly
   */
  this.issue = {/*this code is hidden*/};
  /**
   * The dimentions of an image file. For an image file, the value is
   * `rw=%width%&rh=%heigth%`, for a non-image file the value is `empty`.
   * @name IssueAttachment#metaData
   * @type {String}
   * @readonly
   */
  this.metaData = '';
  /**
   * The group for which the attachment is visible when the visibility is restricted to a single group.
   * @name IssueAttachment#permittedGroup
   * @type {UserGroup}
   */
  this.permittedGroup = {/*this code is hidden*/};
  /**
   * The groups for which the issue is visible when the visibility is restricted to multiple groups.
   * @name IssueAttachment#permittedGroups
   * @type {Set<UserGroup>}
   */
  this.permittedGroups = {/*this code is hidden*/};
  /**
   * The list of users for whom the attachment is visible.
   * @name IssueAttachment#permittedUsers
   * @type {Set<User>}
   */
  this.permittedUsers = {/*this code is hidden*/};
  /**
   * The date and time the attachment was last updated as a timestamp.
   * @name IssueAttachment#updated
   * @type {Number}
   * @readonly
   */
  this.updated = 0;
};
IssueAttachment.prototype = Object.create(PersistentFile.prototype);
IssueAttachment.prototype.constructor = PersistentFile;

/**
 * Permanently deletes the attachment.
 * @since 2018.1.40030
 */
IssueAttachment.prototype.delete = function () {};


/**
 * @classdesc Represents a comment that is added to an issue.
 * @augments BaseComment
 */
var IssueComment = function () {
  /**
   * The absolute URL (permalink) that points to the comment.
   * @example
   * user.notify('Somebody has written something', 'Have a look: ' + comment.url);
   * @name IssueComment#url
   * @type {String}
   * @readonly
   */
  this.url = '';
  /**
   * The user who created the comment.
   * @name IssueComment#author
   * @type {User}
   * @readonly
   */
  this.author = {/*this code is hidden*/};
  /**
   * Time the comment was created.
   * @name IssueComment#created
   * @type {Number}
   * @readonly
   */
  this.created = 0;
  /**
   * The issue the comment belongs to.
   * @name IssueComment#issue
   * @type {Issue}
   * @readonly
   */
  this.issue = {/*this code is hidden*/};
  /**
   * A group whos members are allowed to access the comment.
   * @name IssueComment#permittedGroup
   * @type {UserGroup}
   */
  this.permittedGroup = {/*this code is hidden*/};
  /**
   * Groups whos members are allowed to access the comment.
   * @name IssueComment#permittedGroups
   * @type {Set<UserGroup>}
   */
  this.permittedGroups = {/*this code is hidden*/};
  /**
   * Users that are allowed to access the comment.
   * @name IssueComment#permittedUsers
   * @type {Set<User>}
   */
  this.permittedUsers = {/*this code is hidden*/};
  /**
   * The text of the comment.
   * @name IssueComment#text
   * @type {String}
   */
  this.text = '';
  /**
   * Time the comment was last updated.
   * @name IssueComment#updated
   * @type {Number}
   * @readonly
   */
  this.updated = 0;
  /**
   * The user who last updated the comment.
   * @name IssueComment#updatedBy
   * @type {User}
   * @readonly
   */
  this.updatedBy = {/*this code is hidden*/};
};
IssueComment.prototype = Object.create(BaseComment.prototype);
IssueComment.prototype.constructor = BaseComment;

/**
 * Logically deletes the comment. This means that the comment is marked as deleted, but remains in the database.
 * Users with sufficient permissions can restore the comment or delete the comment permanently from the user interface.
 * The option to delete comments permanently has not been implemented in this API.
 * @since 2018.1.38923
 */
IssueComment.prototype.delete = function () {};


/**
 * @classdesc Represents an issue link type.
 * @augments BaseEntity
 */
var IssueLinkPrototype = function () {
  /**
   * The inward name of the issue link type.
   * @name IssueLinkPrototype#inward
   * @type {String}
   * @readonly
   */
  this.inward = '';
  /**
   * The outward name of the issue link type.
   * @name IssueLinkPrototype#outward
   * @type {String}
   * @readonly
   */
  this.outward = '';
};
IssueLinkPrototype.prototype = Object.create(BaseEntity.prototype);
IssueLinkPrototype.prototype.constructor = BaseEntity;

/**
 * Finds an issue link type by its inward or outward name.
 * @static
 * @param {String} name The inward or outward name of an issue link type.
 *
 * @returns {IssueLinkPrototype} The issue link type.
 */
IssueLinkPrototype.findByName = function (name) {
  return {/*this code is hidden*/};
};


/**
 * @classdesc Represents a tag.
 * @augments WatchFolder
 */
var IssueTag = function () {
  /**
   * The user who created the tag.
   * @name IssueTag#owner
   * @type {User}
   * @readonly
   */
  this.owner = {/*this code is hidden*/};
};
IssueTag.prototype = Object.create(WatchFolder.prototype);
IssueTag.prototype.constructor = WatchFolder;

/**
 * Finds a list of tags with the specified name. The list only includes tags that are visible to the current user.
 * The tags that were created by the current user are returned at the top of the list.
 * "Star" tag is excluded from the results.
 * @static
 * @param {String} name The name of the tag to search for.
 *
 * @returns {Set<IssueTag>} A list of tags that match the specified name.
 */
IssueTag.findByName = function (name) {
  return {/*this code is hidden*/};
};

/**
 * Finds the most relevant tag with the specified name that is visible to the current user.
 * "Star" tag is excluded from the results.
 * @static
 * @param {String} name The name of the tag to search for.
 *
 * @returns {IssueTag} The most relevant tag.
 */
IssueTag.findTagByName = function (name) {
  return {/*this code is hidden*/};
};


/**
 * @classdesc Represents a work item that has been added to an issue.
 * @augments BaseWorkItem
 */
var IssueWorkItem = function () {
  /**
   * The date and time that is assigned to the work item. Stored as a Unix timestamp in UTC. The time part is set to midnight for the current date.
   * @name IssueWorkItem#date
   * @type {Number}
   * @readonly
   */
  this.date = 0;
  /**
   * The duration of the work item in minutes.
   * Writable since 2018.1.40800
   * @name IssueWorkItem#duration
   * @type {Number}
   */
  this.duration = 0;
};
IssueWorkItem.prototype = Object.create(BaseWorkItem.prototype);
IssueWorkItem.prototype.constructor = BaseWorkItem;

/**
 * Permanently deletes the work item.
 * @since 2018.2.42312
 */
IssueWorkItem.prototype.delete = function () {};


/**
 * @classdesc Represents a value in a custom field that has a user associated with it, a so-called owner.
 * @augments Field
 */
var OwnedField = function () {
  /**
   * Field type. Used when defining rule requirements.
   * @name OwnedField.fieldType
   * @type {String}
   * @readonly
   */
  this.fieldType = '';
  /**
   * The user who is associated with the value.
   * @name OwnedField#owner
   * @type {User}
   * @readonly
   */
  this.owner = {/*this code is hidden*/};
};
OwnedField.prototype = Object.create(Field.prototype);
OwnedField.prototype.constructor = Field;


/**
 * @classdesc Represents a custom field in a project that stores a value as a period type.
 * We use org.joda.time.Period as a base class for period values.
 * While you can read the class documentation at
 * http://joda-time.sourceforge.net/apidocs/org/joda/time/Period.html,
 * please note that we support only class members which use the Period class and
 * primitive types like String and int.
 * @example
 * // to convert period to minutes (or other units) use get* methods:
 * var period = issue.fields.Estimation;
 * var minutes = !period ? 0 : (period.getMinutes() +
 *                              60 * (period.getHours() +
 *                                    8 * period.getDays()));
 * // to create Period instance, use toPeriod function from date-time module:
 * issue.fields.Estimation = dateTime.toPeriod(3 * 3600 * 1000); // 3h in ms
 * issue.fields.Estimation = dateTime.toPeriod('3h'); // short form
 * issue.fields.Estimation = dateTime.toPeriod('2w4d3h15m'); // full form
 * @augments SimpleProjectCustomField
 */
var PeriodProjectCustomField = function () {
};
PeriodProjectCustomField.prototype = Object.create(SimpleProjectCustomField.prototype);
PeriodProjectCustomField.prototype.constructor = SimpleProjectCustomField;


/**
 * @classdesc TODO class documentation missing
 * @augments BaseEntity
 */
var PersistentFile = function () {
  /**
   * The extension that defines the file type.
   * @name PersistentFile#extension
   * @type {String}
   * @readonly
   */
  this.extension = '';
  /**
   * The name of the file.
   * @name PersistentFile#name
   * @type {String}
   * @readonly
   */
  this.name = '';
  /**
   * The size of the attached file in bytes.
   * @name PersistentFile#size
   * @type {Number}
   * @readonly
   */
  this.size = 0;
};
PersistentFile.prototype = Object.create(BaseEntity.prototype);
PersistentFile.prototype.constructor = BaseEntity;


/**
 * @classdesc Represents a YouTrack project.
 * @augments BaseEntity
 */
var Project = function () {
  /**
   * The ID of the project. Use instead of project.shortName, which is deprecated.
   * @name Project#key
   * @type {String}
   * @readonly
   */
  this.key = '';
  /**
   * The name of the project.
   * @name Project#name
   * @type {String}
   * @readonly
   */
  this.name = '';
  /**
   * The email address that is used to send notifications for the project.
   * If a 'From' address is not set for the project, the default 'From' address for the YouTrack server is returned.
   * @example
   * if (issue.becomesReported) {
   *   var lastRelatedEmails = issue.fields['Last message related emails'];
   *   if (lastRelatedEmails) {
   *     lastRelatedEmails.split(' ').forEach(function (email) {
   *       if (email && email.equalsIgnoreCase(issue.project.notificationEmail)) {
   *         var allRelatedEmails = issue.fields['All related emails'];
   *         if (!allRelatedEmails) {
   *           issue.fields['All related emails'] = email;
   *         } else if (!(allRelatedEmails.split(' ').has(email))) {
   *           issue.fields['All related emails'] = allRelatedEmails + ' ' + email;
   *         }
   *       }
   *     });
   *     issue.fields['Last message related emails'] = null;
   *   }
   * }
   * @name Project#notificationEmail
   * @type {String}
   * @readonly
   */
  this.notificationEmail = '';
  /**
   * A UserGroup object that contains the users and members of groups who are assigned to the project team.
   * @name Project#team
   * @type {UserGroup}
   * @readonly
   * @since 2017.4.38235
   */
  this.team = {/*this code is hidden*/};
  /**
   * If the project is currently archived, this property is `true`.
   * @name Project#isArchived
   * @type {Boolean}
   * @readonly
   */
  this.isArchived = true;
  /**
   * The description of the project as shown on the project profile page.
   * @name Project#description
   * @type {String}
   * @readonly
   */
  this.description = '';
  /**
   * The set of custom fields that are available in the project.
   * @name Project#fields
   * @type {Set<ProjectCustomField>}
   * @readonly
   */
  this.fields = {/*this code is hidden*/};
  /**
   * A list of all issues that belong to the project.
   * @name Project#issues
   * @type {Set<Issue>}
   * @readonly
   */
  this.issues = {/*this code is hidden*/};
  /**
   * The user who is set as the project lead.
   * @name Project#leader
   * @type {User}
   * @readonly
   */
  this.leader = {/*this code is hidden*/};
};
Project.prototype = Object.create(BaseEntity.prototype);
Project.prototype.constructor = BaseEntity;

/**
 * Returns the custom field in the project with the specified name.
 * @param {String} name The name of the custom field.
 *
 * @returns {ProjectCustomField} The custom field with the specified name.
 */
Project.prototype.findFieldByName = function (name) {
  return {/*this code is hidden*/};
};

/**
 * Gets the number of minutes that occurred during working hours in a specified interval.
 * For example, if the interval is two days and the number of working hours in a day is set to 8, the result is 2 * 8 * 60 = 960
 * @param {Number} start Start of the interval.
 * @param {Number} end End of the interval.
 *
 * @returns {Number} The number of minutes that occurred during working hours in the specified interval.
 */
Project.prototype.intervalToWorkingMinutes = function (start, end) {
  return 0;
};

/**
 * Finds a project by ID.
 * @static
 * @param {String} key The ID of the project to search for.
 *
 * @returns {Project} The project, or null when there are no projects with the specified ID.
 */
Project.findByKey = function (key) {
  return {/*this code is hidden*/};
};

/**
 * Finds a project by name.
 * @static
 * @param {String} name The name of the project to search for.
 *
 * @returns {Project} The project, or null when there are no projects with the specified name.
 */
Project.findByName = function (name) {
  return {/*this code is hidden*/};
};


/**
 * @classdesc Represents a custom field that is available in a project.
 * @augments BaseEntity
 */
var ProjectCustomField = function () {
  /**
   * The localized name of the field.
   * @name ProjectCustomField#localizedName
   * @type {String}
   * @readonly
   */
  this.localizedName = '';
  /**
   * The name of the field.
   * @name ProjectCustomField#name
   * @type {String}
   * @readonly
   */
  this.name = '';
  /**
   * The text that is displayed for this field when it is empty.
   * @name ProjectCustomField#nullValueText
   * @type {String}
   * @readonly
   */
  this.nullValueText = '';
};
ProjectCustomField.prototype = Object.create(BaseEntity.prototype);
ProjectCustomField.prototype.constructor = BaseEntity;

/**
 * Checks if the changes that are applied in the current transaction remove the condition to show the custom field.
 * @since 2018.2.42312
 * @param {Issue} issue The issue for which the condition for showing the field is checked.
 *
 * @returns {Boolean} When `true`, the condition for showing the field is removed in the current transaction.
 */
ProjectCustomField.prototype.becomesInvisibleInIssue = function (issue) {
  return true;
};

/**
 * Checks if the changes that are applied in the current transaction satisfy the condition to show the custom field.
 * @since 2018.2.42312
 * @param {Issue} issue The issue for which the condition for showing the field is checked.
 *
 * @returns {Boolean} When `true`, the condition for showing the field is met in the current transaction.
 */
ProjectCustomField.prototype.becomesVisibleInIssue = function (issue) {
  return true;
};

/**
 * Returns the background color that is used for this field value in the specified issue.
 * Can return `null`, `"white"`, or a hex color presentation.
 * @param {Issue} issue The issue for which the background color is returned.
 *
 * @returns {String} The background color that is used for this field value in the specified issue.
 */
ProjectCustomField.prototype.getBackgroundColor = function (issue) {
  return '';
};

/**
 * Returns the foreground color that is used for this field value in the specified issue.
 * Can return `null`, `"white"`, or a hex color presentation.
 * @param {Issue} issue The issue for which the foreground color is returned.
 *
 * @returns {String} The foreground color that is used for this field value in the specified issue.
 */
ProjectCustomField.prototype.getForegroundColor = function (issue) {
  return '';
};

/**
 * Returns the string presentation of the value that is stored in this field in the specified issue.
 * @param {Issue} issue The issue for which the value presentation is returned.
 *
 * @returns {String} The string presentation of the value.
 */
ProjectCustomField.prototype.getValuePresentation = function (issue) {
  return '';
};

/**
 * Checks if a field is visible in the issue.
 * @since 2018.2.42312
 * @param {Issue} issue The issue for which the condition for showing the field is checked.
 *
 * @returns {Boolean} When `true`, the condition for showing the custom field in the issue has been met. It can also mean that the field is not shown on a conditional basis and is always visible.
 */
ProjectCustomField.prototype.isVisibleInIssue = function (issue) {
  return true;
};


/**
 * @classdesc Represents a value in a custom field that stores a version type.
 * @augments Field
 */
var ProjectVersion = function () {
  /**
   * If the version is archived, this property is `true`.
   * @name ProjectVersion#isArchived
   * @type {Boolean}
   * @readonly
   */
  this.isArchived = true;
  /**
   * If the version is released, this property is `true`.
   * @name ProjectVersion#isReleased
   * @type {Boolean}
   * @readonly
   */
  this.isReleased = true;
  /**
   * Field type. Used when defining rule requirements.
   * @name ProjectVersion.fieldType
   * @type {String}
   * @readonly
   */
  this.fieldType = '';
  /**
   * The release date that is associated with the version.
   * @name ProjectVersion#releaseDate
   * @type {Number}
   * @readonly
   */
  this.releaseDate = 0;
};
ProjectVersion.prototype = Object.create(Field.prototype);
ProjectVersion.prototype.constructor = Field;


/**
 * @classdesc Represents a saved search.
 * @augments WatchFolder
 */
var SavedQuery = function () {
  /**
   * The user who created the saved search.
   * @name SavedQuery#owner
   * @type {User}
   * @readonly
   */
  this.owner = {/*this code is hidden*/};
  /**
   * The search query.
   * @name SavedQuery#query
   * @type {String}
   * @readonly
   */
  this.query = '';
};
SavedQuery.prototype = Object.create(WatchFolder.prototype);
SavedQuery.prototype.constructor = WatchFolder;

/**
 * Finds a list of saved searches with the specified name. The list only includes saved searches that are visible to the current user.
 * The saved searches that were created by the current user are returned at the top of the list.
 * @static
 * @param {String} name The name of the saved search to search for.
 *
 * @returns {Set<SavedQuery>} A list of saved searches that match the specified name.
 */
SavedQuery.findByName = function (name) {
  return {/*this code is hidden*/};
};

/**
 * Finds the most relevant saved search with the specified name that is visible to the current user.
 * @static
 * @param {String} name The name of the saved search to search for.
 *
 * @returns {SavedQuery} The most relevant saved search.
 */
SavedQuery.findQueryByName = function (name) {
  return {/*this code is hidden*/};
};


/**
 * @classdesc Base class for custom fields that store simple values like strings and numbers.
 * @augments ProjectCustomField
 */
var SimpleProjectCustomField = function () {
};
SimpleProjectCustomField.prototype = Object.create(ProjectCustomField.prototype);
SimpleProjectCustomField.prototype.constructor = ProjectCustomField;


/**
 * @classdesc Represents a sprint that is associated with an agile board. Each sprint can include issues from one or more projects.
 * @augments BaseEntity
 */
var Sprint = function () {
  /**
   * The name of the sprint.
   * @name Sprint#name
   * @type {String}
   * @readonly
   */
  this.name = '';
  /**
   * If the sprint is currently archived, this property is `true`.
   * @name Sprint#isArchived
   * @type {Boolean}
   * @readonly
   */
  this.isArchived = true;
  /**
   * The agile board that the sprint belongs to.
   * @name Sprint#agile
   * @type {Agile}
   * @readonly
   */
  this.agile = {/*this code is hidden*/};
  /**
   * The end date of the sprint.
   * @name Sprint#finish
   * @type {Number}
   * @readonly
   */
  this.finish = 0;
  /**
   * The start date of the sprint.
   * @name Sprint#start
   * @type {Number}
   * @readonly
   */
  this.start = 0;
};
Sprint.prototype = Object.create(BaseEntity.prototype);
Sprint.prototype.constructor = BaseEntity;

/**
 * Checks whether the specified issue is represented as a swimlane on the agile board that the sprint belongs to.
 * @param {Issue} issue The issue to check.
 *
 * @returns {Boolean} If the specified issue is represented as a swimlane in the sprint, returns `true`.
 */
Sprint.prototype.isSwimlane = function (issue) {
  return true;
};


/**
 * @classdesc Represents a value in a custom field that stores a state type.
 * @augments Field
 */
var State = function () {
  /**
   * Field type. Used when defining rule requirements.
   * @name State.fieldType
   * @type {String}
   * @readonly
   */
  this.fieldType = '';
  /**
   * If issues in this state are considered to be resolved, ths property is `true`.
   * @name State#isResolved
   * @type {Boolean}
   * @readonly
   */
  this.isResolved = true;
};
State.prototype = Object.create(Field.prototype);
State.prototype.constructor = Field;


/**
 * @classdesc Represents a custom field that stores a string of characters as text. When displayed in an issue, the text is shown as formatted in Markdown.
 * @augments SimpleProjectCustomField
 */
var TextProjectCustomField = function () {
};
TextProjectCustomField.prototype = Object.create(SimpleProjectCustomField.prototype);
TextProjectCustomField.prototype.constructor = SimpleProjectCustomField;


/**
 * @classdesc Represents a user account in YouTrack.
 * @augments BaseEntity
 */
var User = function () {
  /**
   * The email address of the user.
   * @name User#email
   * @type {String}
   * @readonly
   */
  this.email = '';
  /**
   * The full name of the user as seen in their profile.
   * @name User#fullName
   * @type {String}
   * @readonly
   */
  this.fullName = '';
  /**
   * The login of the user.
   * @name User#login
   * @type {String}
   * @readonly
   */
  this.login = '';
  /**
   * User's time zone id.
   * @name User#timeZoneId
   * @type {String}
   * @readonly
   */
  this.timeZoneId = '';
  /**
   * The full name of the user or the login if the full name is not set.
   * @name User#visibleName
   * @type {String}
   * @readonly
   */
  this.visibleName = '';
  /**
   * If the user is currently banned, this property is `true`.
   * @name User#isBanned
   * @type {Boolean}
   * @readonly
   */
  this.isBanned = true;
  /**
   * The current (logged in) user.
   * @name User.current
   * @type {User}
   * @readonly
   */
  this.current = {/*this code is hidden*/};
  /**
   * Field type. Used when defining rule requirements.
   * @name User.fieldType
   * @type {String}
   * @readonly
   */
  this.fieldType = '';
};
User.prototype = Object.create(BaseEntity.prototype);
User.prototype.constructor = BaseEntity;

/**
 * Checks whether the user is able to remove their vote from the specified issue.
 * @param {Issue} issue The issue to check.
 *
 * @returns {Boolean} If the user can vote for the issue, returns `true`.
 */
User.prototype.canUnvoteIssue = function (issue) {
  return true;
};

/**
 * Checks whether the user is able to vote for the specified issue.
 * @param {Issue} issue The issue to check.
 *
 * @returns {Boolean} If the user can vote for the issue, returns `true`.
 */
User.prototype.canVoteIssue = function (issue) {
  return true;
};

/**
 * Returns a tag with the specified name that is shared with but not owned by the user. If such a tag does not exist, a null value is returned.
 * @param {String} name The name of the tag.
 *
 * @returns {IssueTag} The tag.
 */
User.prototype.getSharedTag = function (name) {
  return {/*this code is hidden*/};
};

/**
 * Returns a tag that is visible to the user.
 * @param {String} name The name of the tag.
 * @param {Boolean} createIfNotExists If `true` and the specified tag does not exist or is not visible to the user and the user has permission to create tags, a new tag with the specified name is created.
 *
 * @returns {IssueTag} The tag.
 */
User.prototype.getTag = function (name, createIfNotExists) {
  return {/*this code is hidden*/};
};

/**
 * Checks whether the user is granted the specified role in the specified project. When the project parameter is not specified, checks whether the user has the specified role in any project.
 * @param {String} roleName The name of the role to check for.
 * @param {Project} [project] The project to check for the specified role assignment. If omitted, checks if the user has the global role.
 *
 * @returns {Boolean} If the user is granted the specified role, returns `true`.
 */
User.prototype.hasRole = function (roleName, project) {
  return true;
};

/**
 * Checks whether the user is a member of the specified group.
 * @param {String} groupName The name of the group to check for.
 *
 * @returns {Boolean} If the user is a member of the specified group, returns `true`.
 */
User.prototype.isInGroup = function (groupName) {
  return true;
};

/**
 * @typedef {object} JsonForUserNotify
 * @property {String} subject The subject line of the email notification.
 * @property {String} body The message text of the email notification.
 * @property {Boolean} [ignoreNotifyOnOwnChangesSetting] If `false`, the message is not sent when changes are performed on behalf of the current user. Otherwise, the message is sent anyway.
 * @property {Project} [project] When set, the email address that is used as the 'From' address for the specified project is used to send the message.
 */
/**
 * Sends an email notification to the email address that is set in the user profile.
 * @param {String|JsonForUserNotify} subject The subject line of the email notification. Alternatively, pass a JSON specified by {@link JsonForUserNotify}
 * @param {String} [body] The message text of the email notification.
 * @param {Boolean} [ignoreNotifyOnOwnChangesSetting] If `false`, the message is not sent when changes are performed on behalf of the current user. Otherwise, the message is sent anyway.
 * @param {Project} [project] When set, the email address that is used as the 'From' address for the specified project is used to send the message.
 */
User.prototype.notify = function (subject, body, ignoreNotifyOnOwnChangesSetting, project) {};

/**
 * Sends a notification message over Jabber. Similar to the `notify`
 * method, the message won't be sent on own changes and corresponding flag unchecked.
 * @param {String} message The message text for the Jabber notification.
 */
User.prototype.sendJabber = function (message) {};

/**
 * Sends an email notification to the email address that is set in the user profile. An alias for notify(subject, body, true).
 * @param {String} subject The subject line of the email notification.
 * @param {String} body The message text of the email notification.
 */
User.prototype.sendMail = function (subject, body) {};

/**
 * Removes a vote on behalf of the user from the issue, if allowed.
 * @param {Issue} issue The issue from which the vote is removed.
 */
User.prototype.unvoteIssue = function (issue) {};

/**
 * Removes the current user from the list of watchers for the issue
 * (remove `Star` tag).
 * @param {Issue} issue The issue to from which the user is removed as a watcher.
 */
User.prototype.unwatchIssue = function (issue) {};

/**
 * Adds a vote on behalf of the user to the issue, if allowed.
 * @param {Issue} issue The issue to which the vote is added.
 */
User.prototype.voteIssue = function (issue) {};

/**
 * Adds the current user to the issue as a watcher (add `Star` tag).
 * @param {Issue} issue The issue to which the user is added as a watcher.
 */
User.prototype.watchIssue = function (issue) {};

/**
 * Finds users by email.
 * @static
 * @since 2018.2.41100
 * @param {String} email The email to search for.
 *
 * @returns {Set<User>} Users with the specified email.
 */
User.findByEmail = function (email) {
  return {/*this code is hidden*/};
};

/**
 * Finds a user by login.
 * @static
 * @param {String} login The login of the user account to search for.
 *
 * @returns {User} The specified user, or null when a user with the specified login is not found.
 */
User.findByLogin = function (login) {
  return {/*this code is hidden*/};
};

/**
 * Finds a user by email.
 * @static
 * @since 2018.2.41100
 * @param {String} email The email of the user account to search for.
 *
 * @returns {User} The specified user, or null when a user with the specified email is not found or there are multiple users with the specified email.
 */
User.findUniqueByEmail = function (email) {
  return {/*this code is hidden*/};
};


/**
 * @classdesc Represents a group of users.
 * @augments BaseEntity
 */
var UserGroup = function () {
  /**
   * A list of users who are members of the group.
   * @name UserGroup#users
   * @type {Set<User>}
   * @readonly
   */
  this.users = {/*this code is hidden*/};
  /**
   * If the group is the All Users group, this property is `true`.
   * @name UserGroup#isAllUsersGroup
   * @type {Boolean}
   * @readonly
   */
  this.isAllUsersGroup = true;
  /**
   * If the auto-join option is enabled for the group, this property is `true`.
   * @name UserGroup#isAutoJoin
   * @type {Boolean}
   * @readonly
   */
  this.isAutoJoin = true;
  /**
   * The All Users group.
   * @name UserGroup.allUsersGroup
   * @type {UserGroup}
   * @readonly
   */
  this.allUsersGroup = {/*this code is hidden*/};
  /**
   * Field type. Used when defining rule requirements.
   * @name UserGroup.fieldType
   * @type {String}
   * @readonly
   */
  this.fieldType = '';
  /**
   * The description of the group.
   * @name UserGroup#description
   * @type {String}
   * @readonly
   */
  this.description = '';
  /**
   * The name of the group.
   * @name UserGroup#name
   * @type {String}
   * @readonly
   */
  this.name = '';
};
UserGroup.prototype = Object.create(BaseEntity.prototype);
UserGroup.prototype.constructor = BaseEntity;

/**
 * Sends an email notification to all of the users who are members of the group.
 * @example
 * issue.oldValue('permittedGroup').notifyAllUsers('Visibility has been changed',
 *   'The visibility group for the issue ' + issue.getId() +
 *   ' has been changed to ' + permittedGroup.name);
 * @param {String} subject The subject line of the email notification.
 * @param {String} body The message text of the email notification.
 */
UserGroup.prototype.notifyAllUsers = function (subject, body) {};

/**
 * Finds a group by name.
 * @static
 * @param {String} name The name of the group to search for.
 *
 * @returns {UserGroup} The specified user group, or null when a group with the specified name is not found.
 */
UserGroup.findByName = function (name) {
  return {/*this code is hidden*/};
};


/**
 * @classdesc Represents a custom field in a project that stores values as a user type.
 * @augments ProjectCustomField
 */
var UserProjectCustomField = function () {
  /**
   * The list of available values for the custom field.
   * @name UserProjectCustomField#values
   * @type {Set<User>}
   * @readonly
   */
  this.values = {/*this code is hidden*/};
  /**
   * The default value for the custom field.
   * @name UserProjectCustomField#defaultUsers
   * @type {Set<User>}
   * @readonly
   */
  this.defaultUsers = {/*this code is hidden*/};
};
UserProjectCustomField.prototype = Object.create(ProjectCustomField.prototype);
UserProjectCustomField.prototype.constructor = ProjectCustomField;

/**
 * Returns the value that matches the specified login in a custom field that stores values as a user type.
 * @param {String} login The user login to search for in the set of values for the custom field.
 *
 * @returns {User} The user with the specified login. This user can be set as the value for a field that stores a user type.
 */
UserProjectCustomField.prototype.findValueByLogin = function (login) {
  return {/*this code is hidden*/};
};


/**
 * @classdesc Represents a VCS change that is attached to an issue.
 * @augments BaseEntity
 */
var VcsChange = function () {
  /**
   * The list of change processors that the VCS change can be retrieved from.
   * @name VcsChange#changesProcessors
   * @type {Set<ChangesProcessor>}
   * @readonly
   * @since 2018.1.38923
   */
  this.changesProcessors = {/*this code is hidden*/};
  /**
   * The date when the change was applied, as returned by the VCS.
   * @name VcsChange#created
   * @type {Number}
   * @readonly
   * @since 2018.1.39547
   */
  this.created = 0;
  /**
   * The comment text that was included in the commit message.
   * @name VcsChange#text
   * @type {String}
   * @readonly
   * @since 2018.1.38923
   */
  this.text = '';
  /**
   * The name of the branch that the VCS change was committed to.
   * @name VcsChange#branch
   * @type {String}
   * @readonly
   * @since 2018.1.38923
   */
  this.branch = '';
  /**
   * The date when the change was applied, as returned by the VCS.
   * Use `VcsChange.created` instead.
   * @deprecated 2018.1.39547
   * @name VcsChange#date
   * @type {Number}
   * @readonly
   * @since 2018.1.38923
   */
  this.date = 0;
  /**
   * The date when the VCS change was retrieved from the change processor.
   * @name VcsChange#fetched
   * @type {Number}
   * @readonly
   * @since 2018.1.39547
   */
  this.fetched = 0;
  /**
   * A unique identifier. Used by some CI servers in addition to version.
   * @name VcsChange#id
   * @type {Number}
   * @readonly
   * @since 2018.1.38923
   */
  this.id = 0;
  /**
   * The user who authored the VCS change.
   * @name VcsChange#user
   * @type {User}
   * @readonly
   * @since 2018.1.38923
   */
  this.user = {/*this code is hidden*/};
  /**
   * The name of the change author, as returned by the VCS.
   * @name VcsChange#userName
   * @type {String}
   * @readonly
   * @since 2018.1.38923
   */
  this.userName = '';
  /**
   * The version number of the change. For a Git-based VCS, the revision hash.
   * @name VcsChange#version
   * @type {String}
   * @readonly
   * @since 2018.1.38923
   */
  this.version = '';
};
VcsChange.prototype = Object.create(BaseEntity.prototype);
VcsChange.prototype.constructor = BaseEntity;


/**
 * @classdesc Represents a VCS server.
 * @augments BaseEntity
 */
var VcsServer = function () {
  /**
   * The URL of the VCS server.
   * @name VcsServer#url
   * @type {String}
   * @readonly
   * @since 2018.1.38923
   */
  this.url = '';
};
VcsServer.prototype = Object.create(BaseEntity.prototype);
VcsServer.prototype.constructor = BaseEntity;


/**
 * @classdesc Represents a common ancestor of classes that represent tags and saved searches.
 * @augments BaseEntity
 */
var WatchFolder = function () {
  /**
   * The name of the tag or saved search.
   * @name WatchFolder#name
   * @type {String}
   * @readonly
   */
  this.name = '';
  /**
   * The group of users for whom the tag or saved search is visible.
   * If the tag or the saved search is only visible to its owner, the value for this property is `null`.
   * @name WatchFolder#shareGroup
   * @type {UserGroup}
   * @readonly
   */
  this.shareGroup = {/*this code is hidden*/};
  /**
   * The group of users who are allowed to update the settings for the tag or saved search.
   * If the tag or the saved search can only be updated by its owner, the value for this property is `null`.
   * @name WatchFolder#updateShareGroup
   * @type {UserGroup}
   * @readonly
   */
  this.updateShareGroup = {/*this code is hidden*/};
};
WatchFolder.prototype = Object.create(BaseEntity.prototype);
WatchFolder.prototype.constructor = BaseEntity;


/**
 * @classdesc Represents a work type that can be assigned to a work item.
 * @augments BaseEntity
 */
var WorkItemType = function () {
  /**
   * The name of the work item type.
   * @name WorkItemType#name
   * @type {String}
   * @readonly
   */
  this.name = '';
};
WorkItemType.prototype = Object.create(BaseEntity.prototype);
WorkItemType.prototype.constructor = BaseEntity;

/**
 * Returns the set of work item types that are available in a project.
 * @static
 * @param {Project} project The project for which work item types are returned.
 *
 * @returns {Set<WorkItemType>} The set of available work item types for the specified project.
 */
WorkItemType.findByProject = function (project) {
  return {/*this code is hidden*/};
};


exports.Agile = Agile;
exports.BaseComment = BaseComment;
exports.BaseWorkItem = BaseWorkItem;
exports.Build = Build;
exports.BundleProjectCustomField = BundleProjectCustomField;
exports.ChangesProcessor = ChangesProcessor;
exports.EnumField = EnumField;
exports.Field = Field;
exports.GroupProjectCustomField = GroupProjectCustomField;
exports.Issue = Issue;
exports.IssueAttachment = IssueAttachment;
exports.IssueComment = IssueComment;
exports.IssueLinkPrototype = IssueLinkPrototype;
exports.IssueTag = IssueTag;
exports.IssueWorkItem = IssueWorkItem;
exports.OwnedField = OwnedField;
exports.PeriodProjectCustomField = PeriodProjectCustomField;
exports.PersistentFile = PersistentFile;
exports.Project = Project;
exports.ProjectCustomField = ProjectCustomField;
exports.ProjectVersion = ProjectVersion;
exports.SavedQuery = SavedQuery;
exports.SimpleProjectCustomField = SimpleProjectCustomField;
exports.Sprint = Sprint;
exports.State = State;
exports.TextProjectCustomField = TextProjectCustomField;
exports.User = User;
exports.UserGroup = UserGroup;
exports.UserProjectCustomField = UserProjectCustomField;
exports.VcsChange = VcsChange;
exports.VcsServer = VcsServer;
exports.WatchFolder = WatchFolder;
exports.WorkItemType = WorkItemType;
