var entities = require('@jetbrains/youtrack-scripting-api/entities');
var workflow = require('@jetbrains/youtrack-scripting-api/workflow');
var notifications = require('@jetbrains/youtrack-scripting-api/notifications');

exports.rule = entities.Issue.onChange({
  title: workflow.i18n('Send notifications to all unregistered users'),
  guard: function(ctx) {
    return ctx.issue.comments.added.isNotEmpty();
  },
  action: function(ctx) {
    var issue = ctx.issue;
    var comment = issue.comments.added.first();
    if (comment && comment.permittedGroups.isEmpty() && comment.permittedUsers.isEmpty()) {
      var isBlank = function(str) {
        return !str || str.trim().length === 0;
      };
      var allRelatedEmailsStr = issue.fields.allEmails;
      if (isBlank(allRelatedEmailsStr)) {
        return;
      }
      var allRelatedEmails = allRelatedEmailsStr.split(' ');

      var lastMessageRelatedEmailsStr = issue.fields.lastEmails;
      var lastMessageRelatedEmails = isBlank(lastMessageRelatedEmailsStr) ? [] : lastMessageRelatedEmailsStr.split(' ');

      var fromServiceEmail = issue.project.notificationEmail;
      var isEmailAllowed = function(email) {
        return email && email.length && email.toUpperCase() !== fromServiceEmail.toUpperCase();
      };

      var emailsToNotify;
      if (lastMessageRelatedEmails.length) {
        //case 1: comment from unregistered user
        var newUsersEmails = lastMessageRelatedEmails.filter(function(email) {
          return isEmailAllowed(email) && allRelatedEmails.indexOf(email) === -1;
        });
        if (newUsersEmails.length) {
          issue.fields.allEmails = allRelatedEmails.concat(newUsersEmails).join(' ');
        }
        issue.fields.lastEmails = null;

        emailsToNotify = allRelatedEmails.filter(function(email) {
          return isEmailAllowed(email) && lastMessageRelatedEmails.indexOf(email) === -1;
        });
      } else {
        //case 2: comment from agent
        emailsToNotify = allRelatedEmails.filter(function(email) {
          return isEmailAllowed(email);
        });
      }

      if (emailsToNotify.length) {
        var message = {
          fromName: lastMessageRelatedEmails.length ? lastMessageRelatedEmails[0] : comment.author.fullName,
          toEmails: emailsToNotify,
          subject: createSubject(issue),
          body: lastMessageRelatedEmails.length ? createGeneralMessage(issue) : createInReplyToMessage(issue)
        };
        notifications.sendEmail(message, issue);
      }
    }
  },
  requirements: {
    lastEmails: {
      name: 'Last message related emails',
      type: entities.Field.stringType
    },
    allEmails: {
      name: 'All related emails',
      type: entities.Field.stringType
    }
  }
});

function createGeneralMessage(issue) {
  var text = issue.comments.added.first().text;
  issue.attachments.added.forEach(function(attachment) {
    text = text + '\n[file:' + attachment.name + ']';
  });
  return issue.wikify(text).trim();
}

function createInReplyToMessage(issue) {
  var messageText = createGeneralMessage(issue);
  var addedComments = issue.comments.added;
  var lastVisibleComment;
  issue.comments.forEach(function(comment) {
    if (!addedComments.has(comment) && comment.permittedGroups.isEmpty() && comment.permittedUsers.isEmpty()) {
      lastVisibleComment = comment;
    }
  });
  var quotedText = lastVisibleComment ?
    issue.wikify(lastVisibleComment.text).trim() :
    issue.wikify(issue.description).trim();

  return [
    '<div style="font-family: sans-serif">',
    '  <div style="padding: 10px 10px; font-size: 13px; border-bottom: 1px solid #D4D5D6;">',
    '    ' + messageText,
    '  </div>',
    '  <blockquote type="cite">',
    '    <div style="font-size: 13px; color: #888;">',
    '      ' + workflow.i18n('In reply to:') + '<br><br>' + quotedText,
    '    </div>',
    '  </blockquote>',
    '  <div style="margin: 20px 0 20px 44px; padding: 4px 0 8px 0; color: #888; font-size: 11px; border-top: 1px solid #D4D5D6;">',
    '    ' + workflow.i18n('You have received this message because you are a participant of the conversation in the issue {0}. Sincerely yours, YouTrack', issue.id),
    '  </div>',
    '</div>'
  ].join('\n');
}

function createSubject(issue) {
  var reStr = workflow.i18n('Re:');
  var summary = issue.summary;
  return summary.indexOf(reStr) === 0 ? summary : reStr + ' ' + summary;
}