var entities = require('@jetbrains/youtrack-scripting-api/entities');
var workflow = require('@jetbrains/youtrack-scripting-api/workflow');

exports.rule = entities.Issue.stateMachine({
  title: workflow.i18n('Define transitions for "State" field with Pomodoro timer'),
  fieldName: 'Pomodoro state',
  states: {
    'Not set': {
      initial: true,
      transitions: {
        start: {
          targetState: 'Timer’s running'
        }
      }
    },

    'Timer’s running': {
      onEnter: function(ctx) {
        ctx.issue.fields['Pomodoro interruption'] = null;
        workflow.message(workflow.i18n('25 minutes pomodoro is started.'));
        ctx.issue.fields['Pomodoro countdown'] = 25;
      },
      transitions: {
        interrupt: {
          targetState: 'Not set',
          action: function(ctx) {
            ctx.issue.fields.required(ctx['Pomodoro interruption'], workflow.i18n('Please specify the interruption cause.'));

            ctx.issue.applyCommand(addWorkToday(
              (25 - ctx.issue.fields['Pomodoro countdown']) + 'm',
              workflow.i18n('Pomodoro was interrupted. The cause: \'') +
              ctx.issue.fields['Pomodoro interruption'].name +
              '\'.'
            ));

            ctx.issue.fields['Pomodoro countdown'] = null;
          }
        },
        reminder: {
          targetState: 'Timer finished',
          after: minutes(25)
        }
      }
    },

    'Timer finished': {
      transitions: {
        'take a break': {
          targetState: 'On a break',
          action: function(ctx) {
            workflow.message(workflow.i18n('5 minutes break.'));
            ctx.issue.applyCommand(addWorkToday('25m', '+1 pomodoro.'));
            ctx.issue.fields['Pomodoro countdown'] = 5;
          }
        },
        'discard': {
          targetState: 'Not set',
          action: function(ctx) {
            ctx.issue.fields.required(ctx['Pomodoro interruption'], workflow.i18n('Please specify the interruption cause.'));

            ctx.issue.applyCommand(addWorkToday(
              '25m',
              workflow.i18n('Pomodoro was discarded. The cause: \'') +
              ctx.issue.fields['Pomodoro interruption'].name +
              "'."
            ));

            ctx.issue.fields['Pomodoro countdown'] = null;
          }
        }
      }
    },

    'On a break': {
      transitions: {
        start: {
          targetState: 'Timer’s running',
          action: function(ctx) {
            ctx.issue.applyCommand(addWorkToday(
              (5 - ctx.issue.fields['Pomodoro countdown']) + 'm',
              workflow.i18n('+1 short break.')
            ));
          }
        },
        reminder: {
          targetState: 'Not set',
          after: minutes(5),
          action: function(ctx) {
            ctx.issue.applyCommand(addWorkToday('5m', workflow.i18n('+1 break.')));
          }
        }
      }
    }
  },
  requirements: {
    'Pomodoro interruption': {
      type: entities.EnumField.fieldType
    },
    'Pomodoro countdown': {
      type: entities.Field.integerType
    }
  }
});


function minutes(m) {
  return m * 60 * 1000;
}

function addWorkToday(countdown, message) {
  return 'add work Today ' + countdown + ' ' + message;
}